clear

mtype = [1 4]; % motor types: 1: kinesin-1, 2: kinesin-2, 3: kinesin-3, 4: DDB, 5: dynein
mtot = length(mtype); % number of motors

kBT = 4.1; % k_B*T: Boltzman constant*absolute temp (@25 degree C)
stepsize = 8; % step size of motor



[kin1, kin2, kin3, DDB, dyn] = motor_initialset(kBT, stepsize); % initialization

nruns = 1000; % number of runs

ExF = 0; % external loading

%% Parameter test section
paras = 1:3; % set parameters 
nparas = length(paras);

Test.nonOff = 0; % non-detach simulation
if Test.nonOff == 0
    Test.switch = 0; % DDB switch mode
    Test.SWdiffON = 0; % diffusion for single motor simulation in switch mode
    
    Test.DDBideal = 1; % load-independent detachment rate for DDB
    if Test.DDBideal == 0         
        Test.DDBcatch = 0; % two-state detachment model
    end
    Test.kin1catch = 0; % two-state detachment model for catch-bond for kinesin-1
end

% for testing kin or DDB stepping rate and stall force Test.kinChemMechStep
% & Test.DDBExpVStep should be turned off
Test.kinChemMechStep = 0; % kinesin chemomechanical cycle for stepping from Andreasson 2015
Test.DDBExpVStep = 0; % Stepping rate is an single exponential function from Elshenawy 2019

Test.kinVoverRL_detach = 0; % detachment rate from V/RL for kinesin
% storage for tracking data
x_cargo = cell(nparas, 1);
x_motor = cell(nparas, 1);
timestamp = cell(nparas, 1);

timeFPS = cell(nparas, 1);
x_cargoFPS = cell(nparas, 1);
Vfps = cell(nparas, 1);

%% run Gillespie stochastic algorithm
for ii = 1:nparas
    mtype(1) = paras(ii);
    [x_cargo{ii}, x_motor{ii}, timestamp{ii}] = Motorstepping_Gillespie(kin1, kin2, kin3, DDB, dyn, mtype, kBT, stepsize, nruns, Test);

    figure(1)
    subplot(3, 1, ii)
    [timeFPS{ii}, x_cargoFPS{ii}, Vfps{ii}] = PlotFPS(x_cargo{ii}, timestamp{ii}, nruns, 3.5, 1, 1);
end

%% Initial setting
function[kin1, kin2, kin3, DDB, dyn] = motor_initialset(kBT, stepsize)

kin1.Vel0 = 586; % nm/s, unloaded velocity
kin1.deltaV = 3.6; % 
kin1.L0 = [87 610]; % unloaded runlength
kin1.deltaL = [0.27 2.3]; %
kin1.ksb0 = 3; %
kin1.ksb4 = 1.35; % backstepping rate @4pN
kin1.ksf0 = kin1.Vel0/stepsize + kin1.ksb0; % unloaded forward stepping rate
kin1.Fstall = 6; % stall force
kin1.koff0 = [7.4 kin1.Vel0/kin1.L0(2)]; % unloaded detach rate
kin1.deltaOff = [0.32 0.6]; % characteristic distance for detachment force, under [assisting, hindering] load
kin1.Fdet = kBT./kin1.deltaOff; % detach force
kin1.katt = 100; % reattach rate
kin1.k2 = 2753; % rate constant for ATP hydrolysis cycle
kin1.k3 = 99; %
kin1.kback4 = 1.35; % backstep rate under 4 pN hindering load
kin1.stiff = 0.2; % motor stiffness

kin2.Vel0 = 307; % nm/s
kin2.deltaV = 2.25; %
kin2.L0 = 403; %
kin2.deltaL = [4.7 1.7]; %
kin2.ksb0 = 3; %
kin2.ksb4 = 2.84; %
kin2.ksf0 = kin2.Vel0/stepsize + kin2.ksb0; %
kin2.Fstall = 6; %
kin2.koff0 = [15 kin2.Vel0/kin2.L0]; %
kin2.deltaOff = []; %
kin2.Fdet = 3; %
kin2.katt = 300; %
kin2.k2 = 1469; %
kin2.k3 = 66.8; %
kin2.kback4 = 2.84;
kin2.stiff = 0.2;

kin3.Vel0 = 910; % nm/s
kin3.deltaV = []; %
kin3.L0 = 5540; %
kin3.deltaL = []; %
kin3.ksb0 = 3; %
kin3.ksb4 = []; %
kin3.ksf0 = kin3.Vel0/stepsize + kin3.ksb0; %
kin3.Fstall = 6; %
kin3.koff0 = kin3.Vel0/kin3.L0; %
kin3.deltaOff = []; %
kin3.Fdet = 1.3; %
kin3.katt = 990; %
kin3.stiff = 0.2;

dyn.Vel0 = []; % nm/s
dyn.deltaV = []; %
dyn.L0 = []; %
dyn.deltaL = []; %
dyn.ksb0 = 8.8; %
dyn.ksb4 = []; %
dyn.ksf0 = 35.3; %
dyn.Fstall = 1.25; %
dyn.koff0 = 1; %
dyn.deltaOff = []; %
dyn.Fdet = 0.87; %
dyn.katt = 5; %
dyn.stiff = 0.065;

DDB.Vel0 = 360; % nm/s
DDB.Vmin = -201; % velocity under super stall
DDB.deltaV = 1.5; %
DDB.L0 = []; %
DDB.deltaL = []; %
DDB.ksb0 = 5; %
DDB.ksb4 = []; %
DDB.ksf0 = DDB.Vel0/stepsize + DDB.ksb0; %
DDB.Fstall = 3.6; %
DDB.koff0 = 0.1; %
DDB.deltaOff = []; %
DDB.Fdet = 3; %
DDB.katt = 5; %
DDB.stiff = 0.2;
DDB.ksw = [0 1 0.23; 1.8 0 0.07; 3.9 0.33 0]; % DDB switch mode rates, each row = [to P, to S, to D]
DDB.SWdiff_dt = 0.05; % s, time window based on temporal resolution of EXP.
DDB.SWdiff_D = 20000; % nm^2/s, diffusion coefficient (Qingzhou)

% DDR.Vel0 = [];
% DDR.Fstall = 5.7;

end